/****************************************************************************

	PROGRAM: Generic.c

	PURPOSE: Generic template for Windows applications

	FUNCTIONS:

		WinMain() - calls initialization function, processes message loop
		InitApplication() - initializes window data and registers window
		InitInstance() - saves instance handle and creates main window
		MainWndProc() - processes messages
		About() - processes messages for "About" dialog box

	COMMENTS:

		Windows can have several copies of your application running at the
		same time. The variable hInst keeps track of which instance this
		application is so that processing will be to the correct window.

****************************************************************************/

#include <windows.h>		/* required for all Windows applications */
#include <time.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <string.h>
#include <stdio.h>
#include "generic.h"		/* specific to this program */
#include "gdigit.h"
#include "ioadapt.h"

HANDLE hInst;	/* current instance */
HBRUSH hGreenBrush;
HBRUSH hRedBrush;
HPEN hRedPen;
HPEN hGreenPen;
WORD idTimer;
time_t ltime;
struct tm newtime;
struct tm prevtime;
int alarm_hour;
int alarm_min;
int alarm_mode;

/****************************************************************************

	FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

	PURPOSE: calls initialization function, processes message loop

	COMMENTS:

		Windows recognizes this function by name as the initial entry point
		for the program.  This function calls the application initialization
		routine, if no other instance of the program is running, and always
		calls the instance initialization routine.	It then executes a message
		retrieval and dispatch loop that is the top-level control structure
		for the remainder of execution.  The loop is terminated when a WM_QUIT
		message is received, at which time this function exits the application
		instance by returning the value passed by PostQuitMessage().

		If this function must abort before entering the message loop, it
		returns the conventional value NULL.

****************************************************************************/

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;	/* message */

	if (!hPrevInstance)		/* Other instances of app running? */
		if (!InitApplication(hInstance))	/* Initialize shared things */
			return (FALSE);		/* Exits if unable to initialize */

	/* Perform initializations that apply to a specific instance */
	if (!InitInstance(hInstance, nCmdShow))
		return (FALSE);

	/* Acquire and dispatch messages until a WM_QUIT message is received. */
	while (GetMessage(&msg,		/* message structure */
		NULL,	/* handle of window receiving the message */
		NULL,	/* lowest message to examine	*/
		NULL))	/* highest message to examine	*/
	{
		TranslateMessage(&msg);		/* Translates virtual key codes		  */
		DispatchMessage(&msg);		/* Dispatches message to window		  */
	}
	return (msg.wParam);		/* Returns the value from PostQuitMessage */
}


/****************************************************************************

	FUNCTION: InitApplication(HANDLE)

	PURPOSE: Initializes window data and registers window class

	COMMENTS:

		This function is called at initialization time only if no other
		instances of the application are running.  This function performs
		initialization tasks that can be done once for any number of running
		instances.

		In this case, we initialize a window class by filling out a data
		structure of type WNDCLASS and calling the Windows RegisterClass()
		function. Since all instances of this application use the same window
		class, we only need to do this when the first instance is initialized.


****************************************************************************/

BOOL InitApplication(HANDLE hInstance)
{
	WNDCLASS  wc;

	/* Fill in window class structure with parameters that describe the
		main window. */

	wc.style = CS_DBLCLKS;				/* Class style(s) */
	wc.lpfnWndProc = MainWndProc;	/* Function to retrieve messages for  */
									/* windows of this class */
	wc.cbClsExtra = 0;				/* No per-class extra data */
	wc.cbWndExtra = 0;				/* No per-window extra data */
	wc.hInstance = hInstance;		/* Application that owns the class */
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = GetStockObject(BLACK_BRUSH);
	wc.lpszMenuName =  "AClockMenu";	/* Name of menu resource in .RC file. */
	wc.lpszClassName = "AClockWClass"; /* Name used in call to CreateWindow. */

	/* Register the window class and return success/failure code. */
	return (RegisterClass(&wc));
}

/****************************************************************************

	FUNCTION:  InitInstance(HANDLE, int)

	PURPOSE:  Saves instance handle and creates main window

	COMMENTS:

		This function is called at initialization time for every instance of
		this application.  This function performs initialization tasks that
		cannot be shared by multiple instances.

		In this case, we save the instance handle in a static variable and
		create and display the main program window.
        
****************************************************************************/

BOOL InitInstance(HANDLE hInstance, int nCmdShow)
{
	HWND	hWnd;	/* Main window handle */

    int h_inifile;
    char line[80];
    int i, j;
    int window_width;
    int window_height;
    BOOL b_widthset, b_heightset;
    BOOL b_hourset, b_minset;
    BOOL b_modeset;

    /* Read the width and the height of the window. */
    h_inifile = open("ACLOCK.INI", O_RDONLY | O_TEXT);
    b_widthset = FALSE; b_heightset = FALSE;
    if( h_inifile != -1 )
    {
        while( io_getline(line, h_inifile, 80) != NULL )
        {
            strupr(line);
            if( strstr(line, "WIDTH=") == line )
            {
                if( sscanf(line, "WIDTH=%d", &window_width) )
                {
                    b_widthset = TRUE;
                }
            }
            else if( strstr(line, "HEIGHT=") == line )
            {
                if( sscanf(line, "HEIGHT=%d", &window_height) )
                {
                    b_heightset = TRUE;
                }
            }
            else if( strstr(line, "HOUR=") == line )
            {
                if( sscanf(line, "HOUR=%d", &alarm_hour) )
                {
                    b_hourset = TRUE;
                }
            }
            else if( strstr(line, "MIN=") == line )
            {
                if( sscanf(line, "MIN=%d", &alarm_min) )
                {
                    b_minset = TRUE;
                }
            }
            else if( strstr(line, "MODE=") == line )
            {
                if( sscanf(line, "MODE=%d", &alarm_mode) )
                {
                    b_modeset = TRUE;
                }
            }
        }
        close(h_inifile);
    }
    if( !b_widthset || window_width < 0 ) window_width = 450;
    if( !b_heightset || window_height < 0 ) window_height = 200;
    if( !b_hourset || !b_minset || !b_modeset ) alarm_mode = 0;
    if( alarm_mode < 0 || alarm_mode > 2 )
    {
        alarm_mode = 0;
    }
    if( alarm_mode )
    {
        if( alarm_hour < 0 || alarm_hour > 23 || alarm_min < 0 || alarm_min > 59 )
        {
            alarm_mode = 0;
        }
    }

	/* Save the instance handle in static variable, which will be used in
	   many subsequent calls from this application to Windows.	*/

	hInst = hInstance;

	/* Create a main window for this application instance.	*/

	hWnd = CreateWindow(
		"AClockWClass",	/* See RegisterClass() call */
		"Alarm Clock Application",	/* Text for window title bar */
		WS_OVERLAPPEDWINDOW,	/* Window style */
		CW_USEDEFAULT,	/* Default horizontal position */
		CW_USEDEFAULT,	/* Default vertical position */
		window_width,	/* Default width */
		window_height,	/* Default height */
		NULL,	/* Overlapped windows have no parent */
		NULL,	/* Use the window class menu */
		hInstance,	/* This instance owns this window */
		NULL	/* Pointer not needed */
    );

	/* If window could not be created, return "failure" */

	if (!hWnd)
		return (FALSE);

	/* Make the window visible; update its client area; and return "success" */

	ShowWindow(hWnd, nCmdShow);		/* Show the window */
	UpdateWindow(hWnd); 	/* Sends WM_PAINT message */
	return (TRUE);	/* Returns the value from PostQuitMessage */
}

/****************************************************************************

	FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

	PURPOSE:  Processes messages

	MESSAGES:

		WM_COMMAND	  - application menu (About dialog box)
		WM_DESTROY	  - destroy window

	COMMENTS:

		To process the IDM_ABOUT message, call MakeProcInstance() to get the
		current instance address of the About() function.  Then call Dialog
		box which will create the box according to the information in your
		generic.rc file and turn control over to the About() function.	When
		it returns, free the intance address.

****************************************************************************/

LONG FAR PASCAL MainWndProc(HWND hWnd, unsigned message, WORD wParam, LONG lParam)
{
	FARPROC lpProcAbout;		  /* pointer to the "About" function */
    FARPROC lpProcSetAlarm;
    HDC hDC;
    PAINTSTRUCT ps;
    HBRUSH hOldBrush;
    HPEN hOldPen;
    HBRUSH hBlackBrush;
    int digitx, digity;
    DWORD dextent;
    int d;
    char caption[80];

    int h_inifile;
    char line[80];
    RECT rcWindow;

	switch (message)
	{
        case WM_CREATE:
            hGreenBrush = CreateSolidBrush(RGB(0, 127, 0));
            hRedBrush = CreateSolidBrush(RGB(255, 0, 0));
            hGreenPen = CreatePen(PS_SOLID, 1, RGB(0, 127, 0));
            hRedPen = CreatePen(PS_SOLID, 1, RGB(255, 0, 0));
			idTimer = SetTimer(hWnd, NULL, 1000, (FARPROC) NULL);
            if( !idTimer ) DestroyWindow(hWnd);
            time(&ltime);
            newtime = *localtime(&ltime);
            prevtime = newtime;
            break;

		case WM_COMMAND:	/* message: command from application menu */
			if (wParam == IDM_ABOUT)
			{
				lpProcAbout = MakeProcInstance(About, hInst);

				DialogBox(hInst,		 /* current instance		 */
					"AboutBox",			 /* resource to use		 */
					hWnd,			 /* parent handle		 */
					lpProcAbout);		 /* About() instance address */

				FreeProcInstance(lpProcAbout);
				break;
			}
			else	/* Lets Windows process it		 */
				return (DefWindowProc(hWnd, message, wParam, lParam));

		case WM_LBUTTONDBLCLK:
            lpProcSetAlarm = MakeProcInstance(SetAlarm, hInst);
            DialogBox(hInst, "SetAlarmBox", hWnd, lpProcSetAlarm);
            FreeProcInstance(lpProcSetAlarm);
            break;

        case WM_KEYDOWN:
            if( wParam == VK_RETURN )
            {
                lpProcSetAlarm = MakeProcInstance(SetAlarm, hInst);
                DialogBox(hInst, "SetAlarmBox", hWnd, lpProcSetAlarm);
                FreeProcInstance(lpProcSetAlarm);
            }
            break;

        case WM_PAINT:
            hDC = BeginPaint(hWnd, &ps);
            if( !IsIconic(hWnd) )
            {
                hBlackBrush = GetStockObject(BLACK_BRUSH);
                digitx = 20; digity = 20;
                dextent = getdigitextent();
                d = newtime.tm_hour / 10;
                drawdigit(hDC, d, digitx, digity, hGreenBrush);
                digitx += LOWORD(dextent);
                d = newtime.tm_hour % 10;
                drawdigit(hDC, d, digitx, digity, hGreenBrush);
                digitx += LOWORD(dextent);
                drawcolon(hDC, digitx, digity, hGreenBrush, hGreenPen);
                digitx += LOWORD(dextent);
                d = newtime.tm_min / 10;
                drawdigit(hDC, d, digitx, digity, hGreenBrush);
                digitx += LOWORD(dextent);
                d = newtime.tm_min % 10;
                drawdigit(hDC, d, digitx, digity, hGreenBrush);
            }
            EndPaint(hWnd, &ps);
            break;

        case WM_TIMER:
            time(&ltime);
            newtime = *localtime(&ltime);
            if( newtime.tm_min != prevtime.tm_min )
            {
                InvalidateRect(hWnd, NULL, TRUE);
            }
            if( newtime.tm_sec != prevtime.tm_sec )
            {
                prevtime = newtime;
                if( alarm_mode )
                {
                    if( newtime.tm_hour == alarm_hour && newtime.tm_min == alarm_min )
                    {
                        if( 0 <= newtime.tm_sec && newtime.tm_sec <= 19  )
                        {
                            MessageBeep(0);
                        }
                        else
                        {
                            if( alarm_mode == 1 )
                            {
                                alarm_mode = 0;
                            }
                        }
                    }
                }
            }
            break;

		case WM_DESTROY:		  /* message: window being destroyed */
            DeleteObject(hGreenBrush);
            DeleteObject(hRedBrush);
            DeleteObject(hGreenPen);
            DeleteObject(hRedPen);
            if( idTimer ) KillTimer(hWnd, idTimer);
            if( ( h_inifile = open("ACLOCK.INI", O_WRONLY | O_CREAT | O_TRUNC | O_TEXT, S_IREAD | S_IWRITE) ) != -1 )
            {
                if( alarm_mode )
                {
                    wsprintf(line, "HOUR=%d\n", alarm_hour);
                    write(h_inifile, line, strlen(line));
                    wsprintf(line, "MIN=%d\n", alarm_min);
                    write(h_inifile, line, strlen(line));
                }
                wsprintf(line, "MODE=%d\n", alarm_mode);
                write(h_inifile, line, strlen(line));
                GetWindowRect(hWnd, &rcWindow);
                wsprintf(line, "WIDTH=%d\n", rcWindow.right - rcWindow.left);
                write(h_inifile, line, strlen(line));
                wsprintf(line, "HEIGHT=%d\n", rcWindow.bottom - rcWindow.top);
                write(h_inifile, line, strlen(line));
            }
			PostQuitMessage(0);
			break;

		default:			  /* Passes it on if unproccessed	 */
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return (NULL);
}


/****************************************************************************

	FUNCTION: About(HWND, unsigned, WORD, LONG)

	PURPOSE:  Processes messages for "About" dialog box

	MESSAGES:

		WM_INITDIALOG - initialize dialog box
		WM_COMMAND	  - Input received

	COMMENTS:

		No initialization is needed for this particular dialog box, but TRUE
		must be returned to Windows.

		Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL About(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:		/* message: initialize dialog box */
			return (TRUE);

		case WM_COMMAND:	/* message: received a command */
			if (wParam == IDOK || wParam == IDCANCEL)
			{
				EndDialog(hDlg, TRUE);	/* Exits the dialog box */
				return (TRUE);
			}
			break;
	}
	return (FALSE);		/* Didn't process a message */
}

BOOL FAR PASCAL SetAlarm(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
    int hour, min;
    char buf[80];
    HWND hWnd_almtimeedit;
    HWND hWnd_almonceradio;
    HWND hWnd_almdailyradio;
	switch (message)
	{
		case WM_INITDIALOG:		/* message: initialize dialog box */
            hWnd_almtimeedit = GetDlgItem(hDlg, IDC_ALMTIMEEDIT);
            hWnd_almonceradio = GetDlgItem(hDlg, IDC_ALMONCERADIO);
            hWnd_almdailyradio = GetDlgItem(hDlg, IDC_ALMDAILYRADIO);
            if( alarm_mode == 1 )
            {
                SendMessage(hWnd_almonceradio, BM_SETCHECK, 1, 0L);
            }
            else if( alarm_mode == 2 )
            {
                SendMessage(hWnd_almdailyradio, BM_SETCHECK, 1, 0L);
            }
            if( alarm_mode )
            {
                wsprintf(buf, "%d%d:%d%d", alarm_hour / 10, alarm_hour % 10, alarm_min / 10, alarm_min % 10);
                SetWindowText(hWnd_almtimeedit, buf);
            }
			return (TRUE);

		case WM_COMMAND:	/* message: received a command */
			if (wParam == IDOK || wParam == IDCANCEL)
			{
                if( wParam == IDOK )
                {
                    hWnd_almtimeedit = GetDlgItem(hDlg, IDC_ALMTIMEEDIT);
                    hWnd_almonceradio = GetDlgItem(hDlg, IDC_ALMONCERADIO);
                    hWnd_almdailyradio = GetDlgItem(hDlg, IDC_ALMDAILYRADIO);
                    GetWindowText(hWnd_almtimeedit, buf, 80);
                    if( sscanf(buf, "%d:%d", &hour, &min) == 2 )
                    {
                        if( 0 <= hour && hour <= 23 && 0 <= min && min <= 59 )
                        {
                            alarm_hour = hour;
                            alarm_min = min;
                            if( SendMessage(hWnd_almonceradio, BM_GETCHECK, 0, 0L) )
                            {
                                alarm_mode = 1;
                            }
                            else
                            {
                                alarm_mode = 2;
                            }
                            EndDialog(hDlg, TRUE);
                        }
                        else
                        {
                            MessageBox(hDlg, "The set time is out of range.", "Input Error", MB_ICONINFORMATION);
                        }
                    }
                    else if( sscanf(buf, "%d", &min) == 1 )
                    {
                        // convert the relative time to real time.
                        if( min >= 0 )
                        {
                            alarm_hour = newtime.tm_hour;
                            alarm_min = newtime.tm_min;
                            if( ( alarm_min += min % 60 ) >= 60 )
                            {
                                alarm_hour++;
                                alarm_min -= 60;
                            }
                            if( ( alarm_hour += (min / 60) % 24 ) >= 24 )
                            {
                                alarm_hour -= 24;
                            }
                            if( SendMessage(hWnd_almonceradio, BM_GETCHECK, 0, 0L) )
                            {
                                alarm_mode = 1;
                            }
                            else if( SendMessage(hWnd_almdailyradio, BM_GETCHECK, 0, 0L) )
                            {
                                alarm_mode = 2;
                            }
                            else
                            {
                                MessageBox(hDlg, "Please set the alarm mode.", "Mode Not Set", MB_ICONINFORMATION);
                            }
                            if( alarm_mode )
                            {
                                EndDialog(hDlg, TRUE);
                            }
                        }
                        else
                        {
                            MessageBox(hDlg, "The number of minutes from now shouldn't be nagative.", "Input Error", MB_ICONINFORMATION);
                        }
                    }
                    else if( buf[0] == '\0' )
                    {
                        alarm_mode = 0;
                        EndDialog(hDlg, TRUE);
                    }
                    else
                    {
                        MessageBox(hDlg, "Format error in input.", "Input Error", MB_ICONINFORMATION);
                    }
                }
                else
				    EndDialog(hDlg, TRUE);	/* Exits the dialog box */
				return (TRUE);
			}
			break;
	}
	return (FALSE);		/* Didn't process a message */
}
