/****************************************************************************

    PROGRAM: Generic.c

    PURPOSE: Generic template for Windows applications

    FUNCTIONS:

        WinMain() - calls initialization function, processes message loop
        InitApplication() - initializes window data and registers window
        InitInstance() - saves instance handle and creates main window
        MainWndProc() - processes messages
        About() - processes messages for "About" dialog box

    COMMENTS:

        Windows can have several copies of your application running at the
        same time.  The variable g_hInstance keeps track of which instance this
        application is so that processing will be to the correct window.

****************************************************************************/

#include <windows.h>        /* required for all Windows applications */
#include <fcntl.h>
#include <io.h>
#include <mbstring.h>
#include <stdio.h>
#include <string.h>
#include <sys\stat.h>
#include <sys\types.h>
#include <time.h>
#include "main.h"        /* specific to this program */
#include "gdigit.h"
#include "ioadapt.h"

HINSTANCE g_hInstance;   /* current instance */
HBRUSH g_hbrGreen;
HBRUSH g_hbrRed;
HPEN g_hpenRed;
HPEN g_hpenGreen;
UINT g_idTimer;
time_t g_time;
struct tm g_tmNewTime;
struct tm g_tmPrevTime;
int g_nAlarmHour;
int g_nAlarmMin;
int g_nAlarmMode;

/****************************************************************************

    FUNCTION: WinMain(HINSTANCE, HINSTANCE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.

****************************************************************************/

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;    /* message */

    if( !hPrevInstance )     /* Other instances of app running? */
        if( !InitApplication(hInstance) )    /* Initialize shared things */
            return (1);     /* Exits if unable to initialize */

    /* Perform initializations that apply to a specific instance */
    if( !InitInstance(hInstance, nCmdShow) )
        return (1);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */
    /* FIXME: error handling of GetMessage() */
    while( GetMessage(&msg,     /* message structure */
        NULL,    /* handle of window receiving the message */
        NULL,    /* lowest message to examine */
        NULL) )  /* highest message to examine */
    {
        TranslateMessage(&msg);     /* Translates virtual key codes */
        DispatchMessage(&msg);      /* Dispatches message to window */
    }
    return (msg.wParam);        /* Returns the value from PostQuitMessage */
}

/****************************************************************************

    FUNCTION: InitApplication(HINSTANCE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.

        For 32-bit Windows programs, window classes are by default not shared
        among different instances, so the window class is always registered
        upon running of the application.

****************************************************************************/

BOOL InitApplication(HINSTANCE hInstance)
{
    WNDCLASSEX wcx;

    /* Fill in window class structure with parameters that describe the
       main window. */

    wcx.cbSize = sizeof(wcx);
    /* Class style(s) */
    wcx.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
    wcx.lpfnWndProc = MainWndProc;   /* Function to retrieve messages for */
                                      /* windows of this class */
    wcx.cbClsExtra = 0;              /* No per-class extra data */
    wcx.cbWndExtra = 0;              /* No per-window extra data */
    wcx.hInstance = hInstance;       /* Application that owns the class */
    /* FIXME: error handling of LoadIcon(), LoadCursor() and GetStockObject() */
    wcx.hIcon = (HICON)LoadImage(hInstance, MAKEINTRESOURCE(IDI_THISAPP),
        IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);
    wcx.hCursor = LoadCursor(NULL, IDC_ARROW);
    wcx.hbrBackground = GetStockObject(BLACK_BRUSH);
    /* Name of menu resource in .RC file. */
    wcx.lpszMenuName =  "AClockMenu";
    /* Name used in call to CreateWindow. */
    wcx.lpszClassName = "AClockWClass";
    wcx.hIconSm = (HICON)LoadImage(hInstance, MAKEINTRESOURCE(IDI_THISAPP),
        IMAGE_ICON, 16, 16, LR_DEFAULTCOLOR);

    /* Register the window class and return success/failure code. */
    return (RegisterClassEx(&wcx));
}

/****************************************************************************

    FUNCTION:  InitInstance(HINSTANCE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.
        
****************************************************************************/

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
    HWND hWnd;   /* Main window handle */

    int hIniFile;
    char szLine[80];
    int cxWindowWidth;
    int cyWindowHeight;
    BOOL fWidthSet, fHeightSet;
    BOOL fHourSet, fMinSet;
    BOOL fModeSet;

    /* Read the width and the height of the window */
    /* BEGIN */
    fWidthSet = FALSE; fHeightSet = FALSE;
    fHourSet = FALSE; fMinSet = FALSE;
    fModeSet = FALSE;
    hIniFile = open("AClock.ini", O_RDONLY | O_TEXT);
    if( hIniFile != -1 )
    {
        while( io_GetLine(szLine, hIniFile, 80) != NULL )
        {
            _mbsupr(szLine);
            if( _mbsstr(szLine, "WIDTH=") == szLine )
            {
                if( sscanf(szLine, "WIDTH=%d", &cxWindowWidth) )
                {
                    fWidthSet = TRUE;
                }
            }
            else if( _mbsstr(szLine, "HEIGHT=") == szLine )
            {
                if( sscanf(szLine, "HEIGHT=%d", &cyWindowHeight) )
                {
                    fHeightSet = TRUE;
                }
            }
            else if( _mbsstr(szLine, "HOUR=") == szLine )
            {
                if( sscanf(szLine, "HOUR=%d", &g_nAlarmHour) )
                {
                    fHourSet = TRUE;
                }
            }
            else if( _mbsstr(szLine, "MIN=") == szLine )
            {
                if( sscanf(szLine, "MIN=%d", &g_nAlarmMin) )
                {
                    fMinSet = TRUE;
                }
            }
            else if( _mbsstr(szLine, "MODE=") == szLine )
            {
                if( sscanf(szLine, "MODE=%d", &g_nAlarmMode) )
                {
                    fModeSet = TRUE;
                }
            }
        }
        close(hIniFile);
    }
    /* END */
    if( !fWidthSet || cxWindowWidth < 0 ) cxWindowWidth = 450;
    if( !fHeightSet || cyWindowHeight < 0 ) cyWindowHeight = 200;
    if( !fHourSet || !fMinSet || !fModeSet ) g_nAlarmMode = 0;
    if( g_nAlarmMode < 0 || g_nAlarmMode > 2 )
    {
        g_nAlarmMode = 0;
    }
    if( g_nAlarmMode )
    {
        if( g_nAlarmHour < 0 || g_nAlarmHour > 23 || g_nAlarmMin < 0 || g_nAlarmMin > 59 )
        {
            g_nAlarmMode = 0;
        }
    }

    /* Save the instance handle in static variable, which will be used in
       many subsequent calls from this application to Windows.  */

    g_hInstance = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "AClockWClass", /* See RegisterClass() call */
        "Alarm Clock Application",  /* Text for window title bar */
        WS_OVERLAPPEDWINDOW,    /* Window style */
        CW_USEDEFAULT,   /* Default horizontal position */
        CW_USEDEFAULT,   /* Default vertical position */
        cxWindowWidth,   /* Default width */
        cyWindowHeight,  /* Default height */
        NULL,   /* Overlapped windows have no parent */
        NULL,   /* Use the window class menu */
        hInstance,  /* This instance owns this window */
        NULL    /* Pointer not needed */
    );

    /* If window could not be created, return "failure" */

    if( !hWnd )
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    /* FIXME: error handling of ShowWindow() and UpdateWindow() */
    ShowWindow(hWnd, nCmdShow);     /* Show the window */
    UpdateWindow(hWnd);     /* Sends WM_PAINT message */
    return (TRUE);
}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

        WM_COMMAND    - application menu (About dialog box)
        WM_DESTROY    - destroy window

    COMMENTS:

        To process the IDM_ABOUT message, call MakeProcInstance() to get the
        current instance address of the About() function.  Then call
        DialogBox() which will create the box according to the information in
        your generic.rc file and turn control over to the About() function.
        When it returns, free the intance address.

        32-bit Windows programming doesn't use MakeProcInstance().  Call
        DialogBox() directly.

****************************************************************************/

LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    /* test for stack aggregation */
    /* begin */
    char szBuf[20];
    /* end */
    HDC hDC;
    PAINTSTRUCT ps;
    HBRUSH hbrOld;
    HPEN hpenOld;
    HBRUSH hbrBlack;

    int nDigitX, nDigitY;
    DWORD dwExtent; /* extent of the graphical digit */
    int d; /* the digit */

    int hIniFile;
    char szLine[80];
    RECT rcWindow;

    switch( message )
    {
        case WM_CREATE:
            /* FIXME: error handling of CreateSolidBrush() and other GDI object
             * creation functions */
            g_hbrGreen = CreateSolidBrush(RGB(0, 127, 0));
            g_hbrRed = CreateSolidBrush(RGB(255, 0, 0));
            g_hpenGreen = CreatePen(PS_SOLID, 1, RGB(0, 127, 0));
            g_hpenRed = CreatePen(PS_SOLID, 1, RGB(255, 0, 0));
            g_idTimer = SetTimer(hWnd, 0, 1000, NULL);
            if( !g_idTimer ) DestroyWindow(hWnd);
            time(&g_time);
            /* Warning.  If this program is modified to be multi-threading, the
             * localtime() call is not thread-safe */
            /* NOTE: we copy the structure, not just the pointer */
            g_tmNewTime = *localtime(&g_time);
            g_tmPrevTime = g_tmNewTime;
            break;

        case WM_COMMAND:    /* message: command from application menu */
            if( LOWORD(wParam) == IDM_ABOUT )
            {
                DialogBox(g_hInstance,   /* current instance         */
                    "AboutBox",          /* resource to use      */
                    hWnd,                /* parent handle        */
                    About);              /* About() instance address */
                break;
            }
            else    /* Lets Windows process it       */
                return (DefWindowProc(hWnd, message, wParam, lParam));

        case WM_LBUTTONDBLCLK:
            DialogBox(g_hInstance, "SetAlarmBox", hWnd, SetAlarm);
            break;

        case WM_KEYDOWN:
            if( wParam == VK_RETURN )
            {
                DialogBox(g_hInstance, "SetAlarmBox", hWnd, SetAlarm);
            }
            break;

        case WM_PAINT:
            hDC = BeginPaint(hWnd, &ps);
            if( !IsIconic(hWnd) )
            {
                hbrBlack = GetStockObject(BLACK_BRUSH);
                nDigitX = 20; nDigitY = 20;
                dwExtent = GetDigitExtent();
                d = g_tmNewTime.tm_hour / 10;
                DrawDigit(hDC, d, nDigitX, nDigitY, g_hbrGreen);
                nDigitX += LOWORD(dwExtent);
                d = g_tmNewTime.tm_hour % 10;
                DrawDigit(hDC, d, nDigitX, nDigitY, g_hbrGreen);
                nDigitX += LOWORD(dwExtent);
                DrawColon(hDC, nDigitX, nDigitY, g_hbrGreen, g_hpenGreen);
                nDigitX += LOWORD(dwExtent);
                d = g_tmNewTime.tm_min / 10;
                DrawDigit(hDC, d, nDigitX, nDigitY, g_hbrGreen);
                nDigitX += LOWORD(dwExtent);
                d = g_tmNewTime.tm_min % 10;
                DrawDigit(hDC, d, nDigitX, nDigitY, g_hbrGreen);
            }
            EndPaint(hWnd, &ps);
            break;

        case WM_TIMER:
            time(&g_time);
            g_tmNewTime = *localtime(&g_time);
            if( g_tmNewTime.tm_min != g_tmPrevTime.tm_min )
            {
                InvalidateRect(hWnd, NULL, TRUE);
            }
            if( g_tmNewTime.tm_sec != g_tmPrevTime.tm_sec )
            {
                g_tmPrevTime = g_tmNewTime;
                if( g_nAlarmMode )
                {
                    if( g_tmNewTime.tm_hour == g_nAlarmHour && g_tmNewTime.tm_min == g_nAlarmMin )
                    {
                        /* time is up; alarm */
                        if( 0 <= g_tmNewTime.tm_sec && g_tmNewTime.tm_sec <= 19  )
                        {
                            /* TODO: test how dialog calls like MessageBox()
                             * affect local variable addresses */
                            //sprintf(szBuf, "%08X", szBuf);
                            //MessageBox(hWnd, szBuf, "Test", MB_OK);
                            MessageBeep(MB_OK);
                            FlashWindow(hWnd, (g_tmNewTime.tm_sec % 2 == 0));
                        }
                        else
                        {
                            /* alarm completed */
                            if( g_nAlarmMode == 1 )
                            {
                                g_nAlarmMode = 0;
                            }
                            /* else pass */
                        }
                    }
                    /* else pass */
                }
            }
            break;

        case WM_DESTROY:          /* message: window being destroyed */
            DeleteObject(g_hbrGreen);
            DeleteObject(g_hbrRed);
            DeleteObject(g_hpenGreen);
            DeleteObject(g_hpenRed);
            if( g_idTimer ) KillTimer(hWnd, g_idTimer);
            /* save the ini file */
            if( ( hIniFile = open("AClock.ini", O_WRONLY | O_CREAT | O_TRUNC | O_TEXT, S_IREAD | S_IWRITE) ) != -1 )
            {
                if( g_nAlarmMode )
                {
                    wsprintf(szLine, "HOUR=%d\n", g_nAlarmHour);
                    write(hIniFile, szLine, strlen(szLine));
                    wsprintf(szLine, "MIN=%d\n", g_nAlarmMin);
                    write(hIniFile, szLine, strlen(szLine));
                }
                wsprintf(szLine, "MODE=%d\n", g_nAlarmMode);
                write(hIniFile, szLine, strlen(szLine));
                GetWindowRect(hWnd, &rcWindow);
                wsprintf(szLine, "WIDTH=%d\n", rcWindow.right - rcWindow.left);
                write(hIniFile, szLine, strlen(szLine));
                wsprintf(szLine, "HEIGHT=%d\n", rcWindow.bottom - rcWindow.top);
                write(hIniFile, szLine, strlen(szLine));
                close(hIniFile);
            }
            PostQuitMessage(0);
            break;

        default:              /* Passes it on if unproccessed    */
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

        WM_INITDIALOG - initialize dialog box
        WM_COMMAND    - Input received

    COMMENTS:

        No initialization is needed for this particular dialog box, but TRUE
        must be returned to Windows.

        Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch( message )
    {
        case WM_INITDIALOG:     /* message: initialize dialog box */
            return (TRUE);

        case WM_COMMAND:    /* message: received a command */
            if( LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL )
            {
                EndDialog(hDlg, TRUE);  /* Exits the dialog box */
                return (TRUE);
            }
            break;
    }
    return (FALSE);     /* Didn't process a message */
}

BOOL CALLBACK SetAlarm(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    int nHour, nMin;
    char szBuffer[80];
    HWND hwndAlmTimeEdit;
    HWND hwndAlmOnceRadio;
    HWND hwndAlmDailyRadio;

    switch( message )
    {
        case WM_INITDIALOG:     /* message: initialize dialog box */
            hwndAlmTimeEdit = GetDlgItem(hDlg, IDC_ALMTIMEEDIT);
            hwndAlmOnceRadio = GetDlgItem(hDlg, IDC_ALMONCERADIO);
            hwndAlmDailyRadio = GetDlgItem(hDlg, IDC_ALMDAILYRADIO);
            if( g_nAlarmMode == 1 )
            {
                SendMessage(hwndAlmOnceRadio, BM_SETCHECK, 1, 0);
            }
            else if( g_nAlarmMode == 2 )
            {
                SendMessage(hwndAlmDailyRadio, BM_SETCHECK, 1, 0);
            }
            if( g_nAlarmMode )
            {
                wsprintf(szBuffer, "%02d:%02d", g_nAlarmHour, g_nAlarmMin);
                SetWindowText(hwndAlmTimeEdit, szBuffer);
            }
            return (TRUE);

        case WM_COMMAND:    /* message: received a command */
            if( LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL )
            {
                if( LOWORD(wParam) == IDOK )
                {
                    hwndAlmTimeEdit = GetDlgItem(hDlg, IDC_ALMTIMEEDIT);
                    hwndAlmOnceRadio = GetDlgItem(hDlg, IDC_ALMONCERADIO);
                    hwndAlmDailyRadio = GetDlgItem(hDlg, IDC_ALMDAILYRADIO);
                    GetWindowText(hwndAlmTimeEdit, szBuffer, 80);
                    if( sscanf(szBuffer, "%d:%d", &nHour, &nMin) == 2 )
                    {
                        if( 0 <= nHour && nHour <= 23 && 0 <= nMin && nMin <= 59 )
                        {
                            g_nAlarmHour = nHour;
                            g_nAlarmMin = nMin;
                            if( SendMessage(hwndAlmOnceRadio, BM_GETCHECK, 0, 0) )
                            {
                                g_nAlarmMode = 1;
                            }
                            else if( SendMessage(hwndAlmDailyRadio, BM_GETCHECK, 0, 0) )
                            {
                                g_nAlarmMode = 2;
                            }
                            else
                            {
                                MessageBox(hDlg, "Please set the alarm mode.", "Mode Not Set", MB_ICONINFORMATION);
                            }
                            if( g_nAlarmMode )
                            {
                                EndDialog(hDlg, TRUE);
                            }
                        }
                        else
                        {
                            MessageBox(hDlg, "The set time is out of range.", "Input Error", MB_ICONINFORMATION);
                        }
                    }
                    else if( sscanf(szBuffer, "%d", &nMin) == 1 )
                    {
                        /* "%d" means relative number of minutes */
                        /* convert relative time to real time */
                        if( nMin >= 0 )
                        {
                            g_nAlarmHour = g_tmNewTime.tm_hour;
                            g_nAlarmMin = g_tmNewTime.tm_min;
                            /* add the minute part */
                            if( ( g_nAlarmMin += nMin % 60 ) >= 60 )
                            {
                                g_nAlarmHour++;
                                g_nAlarmMin -= 60;
                                nMin -= nMin % 60;
                            }
                            /* add the remaining hour part, and not more than 24 hours */
                            if( ( g_nAlarmHour += (nMin / 60) % 24 ) >= 24 )
                            {
                                g_nAlarmHour -= 24;
                            }
                            if( SendMessage(hwndAlmOnceRadio, BM_GETCHECK, 0, 0) )
                            {
                                g_nAlarmMode = 1;
                            }
                            else if( SendMessage(hwndAlmDailyRadio, BM_GETCHECK, 0, 0) )
                            {
                                g_nAlarmMode = 2;
                            }
                            else
                            {
                                MessageBox(hDlg, "Please set the alarm mode.", "Mode Not Set", MB_ICONINFORMATION);
                            }
                            if( g_nAlarmMode )
                            {
                                EndDialog(hDlg, TRUE);
                            }
                        }
                        else
                        {
                            MessageBox(hDlg, "The number of minutes from now shouldn't be nagative.", "Input Error", MB_ICONINFORMATION);
                        }
                    }
                    else if( szBuffer[0] == '\0' )
                    {
                        g_nAlarmMode = 0;
                        EndDialog(hDlg, TRUE);
                    }
                    else
                    {
                        MessageBox(hDlg, "Format error in input.", "Input Error", MB_ICONSTOP);
                    }
                }
                else
                    EndDialog(hDlg, TRUE);  /* Exits the dialog box */
                return (TRUE);
            }
            break;
    }
    return (FALSE);     /* Didn't process a message */
}

