# vim: set sts=4 ts=4 sw=4 et nojs fo-=tcqro fo+=tcqro:

# msvc_pragma_pair_check.py:
#     This program works for both Visual C++ (such as in Visual Studio 2022) and Keil uVision,
#     and checks for all "#pragma pack" directives for pairing.
#
#     This program currently aims at C programs. Later it can be extended to support C++
#     programs if needed.

import os
import re
import sys
import traceback
sys.path.append("C:\\cmdtools")
from robbiecommon_py3 import (ClineSwitch, ClineFilter, ListFilesRecursive)

def main():
    switches = [ClineSwitch("p", "path", needarg=True)]
    cl_filter = ClineFilter(switches, sys.argv)
    cl_filter.filter()
    root_path = "."
    if cl_filter.swdict["s:p"].switched:
        root_path = cl_filter.swdict["s:p"].aarg
    lister = ListFilesRecursive(root_path, ["*.c", "*.h"])
    file_list = lister.get_list()
    counter = 0
    for file_path in file_list:
        file_path = os.path.abspath(file_path)
        try:
            process_file(file_path)
        except:
            traceback.print_exc()
            print("At file {}".format(file_path))
        counter += 1
    print("{} files processed".format(counter))
    return 0

def process_file(file_path):
    file1 = None
    try:
        file1 = open(file_path, "rb")
        all_bytes = file1.read()
        file1.close()
        file1 = None
        try:
            all_text = all_bytes.decode("utf-8")
            if all_text.startswith("\ufeff"):
                all_text = all_text.replace("\ufeff", "")
        except:
            try:
                all_text = all_bytes.decode("cp936")
            except:
                raise ValueError("Can't explain file bytes as either UTF-8 or GBK (CP936)")
        process_text(all_text)
    finally:
        if file1 is not None:
            file1.close()
            file1 = None

def process_text(all_text):
    lines = all_text.replace("\r\n", "\n").split("\n")
    if len(lines) >= 1 and lines[-1] == "":
        del lines[-1]
    counter = 0
    ST_NONE = counter
    counter += 1
    ST_DEFAULT_PRAGMA_PACK = counter
    counter += 1
    ST_SPECIAL_PRAGMA_PACK = counter
    counter += 1
    state = ST_DEFAULT_PRAGMA_PACK
    ppack_stack = [] # pragma pack stack
    for line_idx, curr_line in enumerate(lines):
        try:
            ppack_kind = determine_pragma_pack_kind(curr_line)
        except:
            traceback.print_exc()
            print("At line {}".format(line_idx + 1))
        if ppack_kind == PragmaPackConsts.KIND_PRAGMA_PACK_PUSH:
            if state == ST_DEFAULT_PRAGMA_PACK:
                ppack_stack.append("default")
            else: # state == ST_SPECIAL_PRAGMA_PACK
                ppack_stack.append("special")
        elif ppack_kind == PragmaPackConsts.KIND_PRAGMA_PACK_POP:
            if len(ppack_stack) < 1:
                raise ValueError("Pragma pack stack empty when popping. At line {}".format(line_idx
                    + 1))
            if ppack_stack[-1] == "default":
                state = ST_DEFAULT_PRAGMA_PACK
            else: # ppack_stack[-1] == "special"
                state = ST_SPECIAL_PRAGMA_PACK
            del ppack_stack[-1]
        elif ppack_kind == PragmaPackConsts.KIND_PRAGMA_PACK_SET:
            state = ST_SPECIAL_PRAGMA_PACK
        elif ppack_kind == PragmaPackConsts.KIND_PRAGMA_PACK_DEFAULT:
            state = ST_DEFAULT_PRAGMA_PACK
        else: # state == PragmaPackConsts.KIND_NOT_PRAGMA_PACK
            pass
    if state != ST_DEFAULT_PRAGMA_PACK:
        raise ValueError("state not ST_DEFAULT_PRAGMA_PACK at end of file.")
    if len(ppack_stack) != 0:
        raise ValueError("ppack_stack not empty at end of file.")

def determine_pragma_pack_kind(code_line):
    if re.search(r"^\s*#pragma +pack\b.*$", code_line) is not None:
        if re.search(r"^\s*#pragma +pack\(push\).*$", code_line) is not None:
            return PragmaPackConsts.KIND_PRAGMA_PACK_PUSH
        elif re.search(r"^\s*#pragma +pack\(pop\).*$", code_line) is not None:
            return PragmaPackConsts.KIND_PRAGMA_PACK_POP
        elif re.search(r"^\s*#pragma +pack\(\d+\).*$", code_line) is not None:
            return PragmaPackConsts.KIND_PRAGMA_PACK_SET
        elif re.search(r"^\s*#pragma +pack\(\).*$", code_line) is not None:
            return PragmaPackConsts.KIND_PRAGMA_PACK_DEFAULT
        else:
            raise ValueError("Unrecognized pragma pack: {}".format(code_line))
    else:
        return PragmaPackConsts.KIND_NOT_PRAGMA_PACK

class PragmaPackConsts(object):
    counter = 0
    KIND_NOT_PRAGMA_PACK = counter
    counter += 1
    KIND_PRAGMA_PACK_PUSH = counter     # #pragma pack(push)
    counter += 1
    KIND_PRAGMA_PACK_SET = counter      # e.g. #pragma pack(1)
    counter += 1
    KIND_PRAGMA_PACK_POP = counter      # #pragma pack(pop)
    counter += 1
    KIND_PRAGMA_PACK_DEFAULT = counter  # #pragma pack()
    counter += 1
    del counter

if __name__ == "__main__":
    sys.exit(main())
