# vim: set sts=4 ts=4 sw=4 et nojs fo-=tcqro fo+=tcqro fenc=utf-8:

import datetime
import json
import os
import re
import sys

# rm_gitutil_stash_and_switch.py: stashes the current files including untracked
# files and switches to the specified branch.
#
# It is an interactive script.

g_dataFileName = "rm_gitutil_stash_and_switch.dat"
g_dataDirName = ".fantachi"
g_dataFilePath = None

class StashInfo(object):
    def __init__(self):
        self.order = 0
        self.branch_name = ""
        self.comment = ""

    def tojson(self):
        result = []
        result.append("{\"type\": \"StashInfo\", \"order\": ")
        result.append(json.dumps(self.order))
        result.append(", \"branch_name\": ")
        result.append(json.dumps(self.branch_name))
        result.append(", \"comment\": ")
        result.append(json.dumps(self.comment))
        result.append("}")
        return "".join(result)

    def fromjson(self, jsonstr):
        obj = json.loads(jsonstr)
        if "type" not in obj:
            raise ValueError("No type is found in the JSON string. \"StashInfo\" is expected.")
        if obj["type"] != "StashInfo":
            raise ValueError("Invalid type -- {}".format(obj["type"]))
        self.order = int(obj["order"])
        self.branch_name = obj["branch_name"]
        self.comment = obj["comment"]
        return self

    def __str__(self):
        result = []
        result.append("(order: [{}], branch_name: [{}], comment: [{}])".format(self.order,
            self.branch_name, self.comment))
        return "".join(result)

    def __repr__(self):
        return self.tojson()

def main():
    do_things()
    return 0

def do_things():
    status_text = run_cmd_reading_output("git status")
    sys.stdout.write(status_text)
    sys.stdout.flush()
    if re.search(r"^.*?On branch (.*)$", status_text.split("\n")[0].rstrip()) is None:
        print("\"git status\" failed!")
        return
    curr_branch = re.sub(r"^.*?On branch (.*)$", r"\1", status_text.split("\n")[0].rstrip())
    stashls_text = run_cmd_reading_output("git stash list") # stash list text
    #print(stashls_text)
    stashls = parse_stashls_text(stashls_text) # stash list
    print("Branches related to the stashes:")
    branch_names = [x.branch_name for x in stashls]
    branch_names = list(set(branch_names))
    branch_names.sort()
    for name in branch_names:
        print(name)
    ans = get_user_yes_no("Are you sure that we will stash and switch to another branch")
    print(ans)
    if ans != "Y":
        return
    other_branch = ""
    get_data_file_path()
    if os.path.isfile(g_dataFilePath):
        other_branch = load_branch_from_file(g_dataFilePath)
    while True:
        if other_branch == "":
            print("Please enter the target branch name:")
            other_branch = sys.stdin.readline().rstrip()
        print("Current branch name: {}".format(curr_branch))
        print("Target branch name: {}".format(other_branch))
        ans = get_user_yes_no("Are the settings above correct?")
        if ans == "Y":
            break
        else:
            other_branch = ""
    dump_jsonobj_to_file(g_dataFilePath, {"other_branch": other_branch})
    old_stash_name = next((x.comment for x in stashls if x.branch_name == curr_branch),
        "") # the second parameter of next() is the default value
    print("Old stash name: {}".format(old_stash_name))
    new_stash_name = ""
    today_name = datetime.datetime.now().strftime("%Y_%m_%d")
    if re.search(r"^.*\d{4}_\d{2}_\d{2}_\d{3}$", old_stash_name) is not None:
        if re.sub(r"^.*(\d{4}_\d{2}_\d{2})_\d{3}$", r"\1", old_stash_name) == today_name:
            stash_name_num_str = "{:03}".format(int(re.sub(r"^.*_(\d{3})$", r"\1", old_stash_name),
                10) + 1)
            new_stash_name = (re.sub(r"^(.*\d{4}_\d{2}_\d{2}_)\d{3}$", r"\1", old_stash_name)
                + stash_name_num_str)
        else:
            new_stash_name = (re.sub(r"^(.*)\d{4}_\d{2}_\d{2}_\d{3}$", r"\1", old_stash_name)
                + today_name)
    elif re.search(r"^.*\d{4}_\d{2}_\d{2}$", old_stash_name) is not None:
        if re.sub(r"^.*(\d{4}_\d{2}_\d{2})$", r"\1", old_stash_name) == today_name:
            stash_name_num_str = "002"
            new_stash_name = (re.sub(r"^(.*\d{4}_\d{2}_\d{2})$", r"\1", old_stash_name) + "_"
                + stash_name_num_str)
        else:
            new_stash_name = (re.sub(r"^(.*)\d{4}_\d{2}_\d{2}$", r"\1", old_stash_name)
                + today_name)
    else:
        print("Invalid stash name. Please conform to the convention: Description_YYYY_MM_dd or Description_YYYY_MM_dd_nnn")
        return
    print("New stash name: {}".format(new_stash_name))
    run_cmd("git stash push -u -m \"{}\"".format(new_stash_name))
    run_cmd("git status")
    ans = get_user_yes_no("Is the status okay? We are about to switch the branch")
    if ans != "Y":
        return
    run_cmd("git checkout \"{}\"".format(other_branch))
    dump_jsonobj_to_file(g_dataFilePath, {"other_branch": curr_branch})
    run_cmd("git status")
    to_apply_stash_info = next((x for x in stashls if x.branch_name == other_branch),
        None) # the second parameter of next() is the default value
    if to_apply_stash_info is None:
        print("Unable to find the stash to apply. Aborting.")
        return
    to_apply_stash_name = to_apply_stash_info.comment
    print("Will apply the stash {}".format(to_apply_stash_name))
    ans = get_user_yes_no("Is the status okay? We are about to apply the latest stash")
    if ans != "Y":
        return
    # Refresh the stash list {
    stashls_text = run_cmd_reading_output("git stash list") # stash list text
    #print(stashls_text)
    stashls = parse_stashls_text(stashls_text) # stash list
    # } Refresh the stash list
    to_apply_stash_info = next((x for x in stashls if x.branch_name == other_branch),
        None) # the second parameter of next() is the default value
    if to_apply_stash_info is None:
        print("Unable to find the stash to apply. Aborting.")
        return
    to_apply_stash_num = to_apply_stash_info.order
    run_cmd("git stash apply \"stash@{" + str(to_apply_stash_num) + "}\"")
    run_cmd("git status")
    print("Please check if everything is all right.")

def load_branch_from_file(filepath):
    obj = load_json_from_file(filepath)
    return obj["other_branch"]

def load_json_from_file(filepath):
    file1 = None
    try:
        file1 = open(filepath, "r", encoding="utf-8")
        text = file1.read()
        text = text.replace("\uFEFF", "")
        return json.loads(text)
    finally:
        if file1 is not None:
            file1.close()
            file1 = None

def dump_jsonobj_to_file(filepath, obj):
    file1 = None
    try:
        file1 = open(filepath, "w", encoding="utf-8")
        file1.write(json.dumps(obj))
    finally:
        if file1 is not None:
            file1.close()
            file1 = None

def parse_stashls_text(stashls_text):
    lines = [x for x in stashls_text.split("\n") if not x.strip() == ""]
    return [parse_stash_text(x) for x in lines]

def parse_stash_text(stashtext):
    stashtext = stashtext.strip()
    stash_num_part = re.sub(r"^(stash@\{\d+\}).*$", r"\1", stashtext)
    #print(stash_num_part)
    stash_num_sub = re.sub(r"^stash@\{(\d+)\}$", r"\1", stash_num_part)
    #print(stash_num_sub)
    stash_branch_part = re.sub(r": On ([^:]+):.*$", r"\1", stashtext[len(stash_num_part)
        :])
    #print(stash_branch_part)
    stash_comment_part = re.sub(r"^.*: ([^:]+?)$", r"\1", stashtext)
    #print(stash_comment_part)
    return StashInfo().fromjson(json.dumps({"type": "StashInfo", "order": int(stash_num_sub),
        "branch_name": stash_branch_part, "comment": stash_comment_part}))

def run_cmd(cmd):
    print(cmd)
    os.system(cmd)

def run_cmd_reading_output(cmd):
    file1 = None
    try:
        file1 = os.popen(cmd, "r")
        return file1.read()
    finally:
        if file1 is not None:
            file1.close()
            file1 = None

def get_user_yes_no(ques):
    sys.stdout.write(ques + " [YN]? ")
    sys.stdout.flush()
    line = sys.stdin.readline()
    if line.rstrip().lower().startswith("y"):
        return "Y"
    else:
        return "N"

def get_data_file_path():
    global g_dataFilePath
    if g_dataFilePath is None:
        dataDirPath = os.path.join(os.environ["userprofile"], g_dataDirName)
        if not os.path.isdir(dataDirPath):
            os.mkdir(dataDirPath)
        g_dataFilePath = os.path.join(dataDirPath, g_dataFileName)
    return g_dataFilePath

if __name__ == "__main__":
    sys.exit(main())
